#include "msxvideo.h"

#define T32COL 0xF3BF
#define T32CGP 0xF3C1
#define T32ATR 0xF3C3
#define T32PAT 0xF3C5
#define T32NAM 0xF3BD

void screen1_setup_mode() __naked {
	/*
		CHGMOD (005FH)		*3
		Function:	changes the screen mode. The palette is not initialised. To
		initialise it, see CHGMDP in SUB-ROM.
		Input:	A for the screen mode (0 to 8)
		Output:	none
		Registers:	all

		ERAFNK (00CCH)		*1
		Function:	erases the function key display
		Input:	none
		Output:	none
		Registers:	all
	*/

	__asm

	push	ix  	; prologue
	ld	ix,#0
	add	ix,sp

	push af	        ; body
	push bc
	push de
	push hl
	push iy
	ld a,#0x01
	call 0x005F ; --- go to screen 1
	call 0x00CC ; --- erase function keys
	pop iy
	pop hl
	pop de
	pop bc
	pop af

	pop	ix           ;epilogue
	ret

	__endasm;
}

void screen1_setup_sprites(char spritesize, char zoom) __naked {

	/*
		CLRSPR (0069H)		*3
		Function:	initialises all sprites. The sprite pattern is cleared to
		null, the sprite number to the sprite plane number, the
		sprite colour to the foregtound colour. The vertical location
		of the sprite is set to 209 (mode 0 to 3) or 217
		(mode 4 to 8).
		Input:	SCRMOD (FCAFH) for the screen mode
		Output:	none
		Registers:	all

		SETT32 (007BH)		*3
		Function:	set only VDP in GRAPHIC1 mode (32x24)
		Input:	same as INIT32
		Output:	none
		Registers:	all
	*/

	__asm

	push ix  	; prologue
	ld	ix,#0
	add	ix,sp

	push af    ; body
	push bc
	push de
	push hl
	push iy
  
	ld b,#0x00
	ld a,4(ix)
	and #0x0f
	cp #0x08
	jr z,$1
	set 1,b ; --- if 16x16 sprites => set bit 1
$1:
	ld a,5(ix)
	cp #0x00
	jr z, $2
	set 0,b ; --- if zoom sprites => set bit 0
$2:
	ld hl,#0xf3e0 ; --- read vdp(1) from mem
	ld a,(hl)
	and #0xfc
	or b
	ld (hl),a
	call 0x007b ; --- change vdp(1) bits

	ld a,#0x01
	ld hl,#0xfcaf
	ld (hl),a
	call 0x0069 ; --- reset sprite attributes

	pop iy
	pop hl
	pop de
	pop bc
	pop af

	pop	ix           ;epilogue
	ret

	__endasm;
}

void screen_color(char fg, char bg, char bd) __naked {
	/*
		CHGCLR (0062H)		*1
		Function:	changes the screen colour
		Input:	A for the mode
		FORCLR (F3E9H) for foreground color
		BAKCLR (F3EAH) for background color
		BDRCLR (F3EBH) for border colour
		Output:	none
		Registers:	all
	*/
	__asm
	push ix  	; prologue
	ld	ix,#0
	add	ix,sp

	push af    ; body
	push bc
	push de
	push hl
	push iy

	ld hl,#0xf3e9
	ld a,4(ix)
	ld (hl),a

	inc hl
	ld a,5(ix)
	ld (hl),a

	inc hl
	ld a,6(ix)
	ld (hl),a

	call 0x0062 ; --- change color
	pop iy
	pop hl
	pop de
	pop bc
	pop af

	pop	ix           ;epilogue
	ret
	__endasm;
}

char vpeek(int address) __naked {
	/*
		RDVRM (004AH)		*1
		Function:	reads the contents of VRAM. This is for TMS9918, so only the
		14 low order bits of the VRAM address are valid. To use all
		bits, call NRDVRM.
		Input:	HL for VRAM address to be read
		Output:	A for the value which was read
		Registers:	AF
	*/
	__asm
	push ix  	; prologue
	ld	ix,#0
	add	ix,sp

	push af     ; body
	ld l,4(ix)
	ld h,5(ix)
	call 0x004a
	ld h,#0x00
	ld l,a
	pop af

	pop	ix           ;epilogue
	ret
	__endasm;
}

void vpoke(int address, char value) __naked {
	/*
		WRTVRM (004DH)		*1
		Function:	writes data in VRAM. This is for TMS9918, so only the 14 low
		order bits of the VRAM address are valid. To use all bits,
		call NWRVRM.
		Input:	HL for VRAM address, A for data
		Output:	none
		Registers:	AF
	*/

	__asm
	push ix  	; prologue
	ld	ix,#0
	add	ix,sp

	push af ; body
	ld l,4(ix)
	ld h,5(ix)
	ld a,6(ix)
	call 0x004d
	pop af

	pop	ix           ;epilogue
	ret
	__endasm;
}

void vpoke_block(int address, char* block, int size) {
	while (size-- > 0) {
		vpoke(address++, *(block++));
	}
}

void screen1_sprite_profile(char number, unsigned char* profile,
		char spritesize) {
	int mem = peek_word(T32PAT) + (number & 0xff) * (spritesize & 0xff);
	vpoke_block(mem, profile, spritesize);
}

void screen1_sprite_put(char number, char xpos, char ypos, char color) {
	unsigned int mem = peek_word(T32ATR) + (number & 0xff) * 4;
	vpoke(mem + 0, ypos);
	vpoke(mem + 1, xpos);
	vpoke(mem + 3, color);
}

void wait_retrace() __naked {
  __asm

  push af
  di
$3:
  in a,(0x99)
  and #0x80
  cp #0x00
  jr z, $3
  ei
  pop af
  ret

  __endasm;
}

void screen1_setup_chars(){
  int sz,off;

  off=peek_word(T32CGP)+'0'*8;  
  for (sz=0;sz<('z'-'0')*8;sz++){
    vpoke( off, vpeek(off)|vpeek(off)>>1);
    off++;
  }

  off=peek_word(T32COL)+'0'/8;
  vpoke(off++,0xC0);
  vpoke(off++,0xC0);

  off=peek_word(T32COL)+'A'/8;
  vpoke(off++,0x20);
  vpoke(off++,0x20);
  vpoke(off++,0x20);
  vpoke(off++,0x20);

  off=peek_word(T32COL)+'a'/8;
  vpoke(off++,0x20);
  vpoke(off++,0x20);
  vpoke(off++,0x20);
  vpoke(off++,0x20);

  off=peek_word(T32COL)+'\310'/8;
  vpoke(off++,0x50);
  
  off=peek_word(T32CGP)+128*8;
  vpoke(off++, 0x38); // ..XXX...
  vpoke(off++, 0x74); // .XXX.X..
  vpoke(off++, 0xFA); // XXXXX.X.
  vpoke(off++, 0xFE); // XXXXXXX.
  vpoke(off++, 0xFE); // XXXXXXX.
  vpoke(off++, 0x7C); // .XXXXX..
  vpoke(off++, 0x38); // ..XXX...
  vpoke(off++, 0x00); // ........
  off=peek_word(T32COL)+128/8;
  vpoke(off++, 0xA0);

  off=peek_word(T32COL)+160/8;
  vpoke(off, 0x41);

  off=peek_word(T32CGP)+160*8;
  vpoke(off++, 0x7F); // .xxxxxxx
  vpoke(off++, 0x7F); // .xxxxxxx
  vpoke(off++, 0x7F); // .xxxxxxx
  vpoke(off++, 0x7F); // .xxxxxxx
  vpoke(off++, 0x7F); // .xxxxxxx
  vpoke(off++, 0x7F); // .xxxxxxx
  vpoke(off++, 0x55); // .x.x.x.x
  vpoke(off++, 0x00); // ........

  vpoke(off++, 0xFF); // xxxxxxxx
  vpoke(off++, 0xFE); // xxxxxxx.
  vpoke(off++, 0xFF); // xxxxxxxx
  vpoke(off++, 0xFE); // xxxxxxx.
  vpoke(off++, 0xFF); // xxxxxxxx
  vpoke(off++, 0xFE); // xxxxxxx.
  vpoke(off++, 0x55); // .x.x.x.x
  vpoke(off++, 0x00); // ........

}

void screen1_setup_back(){
  int sz,off;

  off=peek_word(T32NAM);
  for (sz=0;sz<16;sz++){
    vpoke(off++, 160); vpoke(off++, 161);
  }  
  off=peek_word(T32NAM)+32*11;
  for (sz=0;sz<16;sz++){
    vpoke(off++, 160); vpoke(off++, 161);
  }  
  off=peek_word(T32NAM)+32*12;
  for (sz=0;sz<16;sz++){
    vpoke(off++, 161); vpoke(off++, 160);
  }  
  off=peek_word(T32NAM)+32*23;
  for (sz=0;sz<16;sz++){
    vpoke(off++, 160); vpoke(off++, 161);
  }  

}

void screen1_back_rotate(char x){
  char tmp1, tmp2;
  int i, off;
  off=peek_word(T32CGP)+160*8;

  switch(x){
    case 1:
      for (i=0;i<8;i++){
        tmp1=vpeek(off+i);
        tmp2=vpeek(off+i+8);
        vpoke( off+i, (tmp1<<1)&0xfe| ((tmp2&0x80)?1:0) );
        vpoke( off+i+8, (tmp2<<1)&0xfe | ((tmp1&0x80)?1:0) );
      }
    break;
    case -1:
      for (i=0;i<8;i++){
        tmp1=vpeek(off+i);
        tmp2=vpeek(off+i+8);
        vpoke( off+i, (tmp1>>1)&0x7f | ((tmp2&0x01)?0x80:0) );
        vpoke( off+i+8, (tmp2>>1)&0x7f | ((tmp1&0x01)?0x80:0) );
      }
    break;
  }  
}

